% Code for: Capital Reallocation and Aggregate Productivity
% R. Cooper & I. Schott
% Code version: September 2022
%
% Moments for the stationary economy

% Compute steady state reallocation, dispersion in MPK and APK
[epsg, kg]  = ndgrid(PARMS.eps, k_sol);

% Firm-level output
q_pol       = epsg.*kg.^(PARMS.ALPHA); % same as c_ + ac_
log_TFPR    = log(q_pol) - PARMS.ALPHA*log(kg);
log_ARPK    = log(epsg.*kg.^(PARMS.ALPHA-1));
R           = abs(kg(:) - kg(:)'); % Total Reallocation
% Multiply every possible reallocation amount with the respective
% transition and then multiply that with the stationary distribution
R_ss        = 1/2 * probst'*sum(trans.*R,2); %

% Compute the output-weighted standard deviation of log TFPR
stdlogtfp   = std(log_TFPR(:),q_pol(:).*probst);

% Compute the output-weighted standard deviation of log APK
stdlogYK    = std(log_ARPK(:),q_pol(:).*probst);

%% Untargeted moments

% Serial correlation of firm-level log(sales)
lS          = log(q_pol(:));
w           = probst.*trans; %
W           = w(w~=0);
[x, y]      = ndgrid(lS,lS');
X           = [ones(size(W)), x(w~=0)]; % include intercept
part1a      = X'.*W'; % Weights from stationary distribution
part1       = part1a*X;
part2       = part1a*y(w~=0);
betahat     = part1\part2;
rho_lS      = betahat(2);

% Correlation between productivity and reallocation
% Weights here indicate the weight in the stationary distribution
epsprime    = kron(ones(PARMS.N*PARMS.N,PARMS.N),PARMS.eps');
corr_r_eps  = weightedcorrs([epsprime(w~=0),R(w~=0)],W);
% Reallocation status
R_pos       = R > 0;
corr_pi_eps = weightedcorrs([epsprime(w~=0),R_pos(w~=0)],W);

% Correlations with ARPK instead of EPS
[x, ~]      = ndgrid(log_ARPK(:),log_ARPK(:)');
corr_r_apk  = weightedcorrs([x(w~=0),R(w~=0)],W);


% Importance of small capital adjustments
% Fraction of adjustments in the lowest 10th of adjustments
x           = R(:);
w           = probst.*trans;
% Exclude zeros, as in the data
idx         = x>0;
x           = x(idx==1);
w           = w(idx);
R_fractions = histwc(x,w/sum(w),10);  

% Autocorrelation of firm-level log(reallocation)
% Run a simulation for this
% If you want to re-run this simulation, set redo = 1
redo    = 0;
if redo == 1 
    I           = 500; % firms
    T           = 1000; % periods
    rng(12345)
    ini         = randsample(1:PARMS.N,I,true,PARMS.PP); % new draw
    kini        = k_sol(ini(:));
    reall       = zeros(I,T);
    TFP         = zeros(I,T);
    for tt = 1:T
        X       = zeros(I,1); % new prod draw
        for ii = 1:I
            X(ii)   = randsample(1:PARMS.N,1,true,PARMS.prob_rho(ini(ii),:));
        end
        knew        = k_sol(X);
        reall(:,tt) = knew - kini;
        TFP(:,tt)   = PARMS.eps(X);
        % update
        ini         = X;
        kini        = knew;
    end
    % Panel regression
    reall(reall>0) = 0; % Set positives to zero to avoid double counting
    reall(reall<0) = reall(reall<0) * (-1); % sales of capital, make positive
    x               = reall(:,1:end-1);
    y               = reall(:,2:end);
    inter           = ones(length(x(:)),1);
    idfe            = repmat((1:I)',1,T-1);
    tfe             = repmat(1:T-1,I,1);
    X               = [inter, x(:), idfe(:), tfe(:)];
    part1           = X'*X;
    part2           = X'*y(:);
    betahat         = part1\part2;
    rho_lR          = betahat(2);
    % save result
    filename        = sprintf('rho_lR_%2.2f.mat', PARMS.ALPHA);
    fullname        = fullfile(PARMS.resfolder, filename);
    save(fullname,'rho_lR');
else
    filename        = sprintf('rho_lR_%2.2f.mat', PARMS.ALPHA);
    fullname        = fullfile(PARMS.resfolder, filename);
    load(fullname,'rho_lR');
end


